/*
 * Google 2FA + App Password Generator
 * By @xsystemaudit
 * Thanks to Ulul Azmi
 * 
 * Note untuk developer:
 * - Jangan Ganti Copyright untuk menghargai kreator
 * - Hargai Orang lain jika anda ingin dihargai
 */

const BrowserManager = require('./core/browser');
const GoogleAuth = require('./core/googleAuth');
const TOTPSetup = require('./core/totpSetup');
const AppPassword = require('./core/appPassword');
const BulkProcessor = require('./core/bulkProcessor');
const logger = require('./utils/logger');
const { delay } = require('./utils/helpers');
const fs = require('fs').promises;
const path = require('path');
const chalk = require('chalk');
const readline = require('readline');

class Google2FABot {
  constructor() {
    this.results = [];
  }

  async processAccount(email, password, appName = '@xsystemaudit', accountNumber = null) {
    const startTime = Date.now();
    const browserManager = new BrowserManager();

    const result = {
      email,
      secretKey: '',
      totpCode: '',
      appPassword: '',
      status: 'failed',
      time: 0,
      message: '',
      twoFAActive: false,
      appPasswordCreated: false
    };

    const prefix = accountNumber ? `[${accountNumber}]` : '';

    try {
      // Write "Processing" without newline
      process.stdout.write(`${prefix} Processing ${email}...`);

      const page = await browserManager.launch();

      const auth = new GoogleAuth(page);
      const totp = new TOTPSetup(page);
      const appPwd = new AppPassword(page);

      const loginSuccess = await auth.login(email, password);
      if (!loginSuccess) {
        throw new Error('Login failed');
      }

      const nav2FASuccess = await totp.navigateTo2FA();
      if (!nav2FASuccess) {
        throw new Error('Navigation failed');
      }

      const secretKey = await totp.setupAuthenticator();
      if (!secretKey) {
        throw new Error('Setup failed');
      }
      result.secretKey = secretKey;

      const activateSuccess = await totp.activate2FA();
      if (activateSuccess) {
        result.twoFAActive = true;
      }

      const appPassword = await appPwd.generate(appName);
      if (appPassword) {
        result.appPassword = appPassword;
        result.appPasswordCreated = true;
      }

      // Clear line and write final status
      readline.clearLine(process.stdout, 0);
      readline.cursorTo(process.stdout, 0);

      if (result.twoFAActive && result.appPasswordCreated) {
        result.status = 'complete';
        result.message = 'success';
        console.log(chalk.green(`${prefix} ${email} : success`));
      } else if (result.secretKey && result.twoFAActive) {
        result.status = 'partial';
        result.message = 'partial (2FA only)';
        console.log(chalk.yellow(`${prefix} ${email} : partial`));
      } else {
        result.status = 'failed';
        result.message = 'failed';
        console.log(chalk.red(`${prefix} ${email} : failed`));
      }

    } catch (error) {
      result.message = error.message;
      // Clear line and write error
      readline.clearLine(process.stdout, 0);
      readline.cursorTo(process.stdout, 0);
      console.log(chalk.red(`${prefix} ${email} : ${error.message}`));
    } finally {
      await browserManager.close();
      result.time = ((Date.now() - startTime) / 1000).toFixed(2);
      this.results.push(result);
    }

    return result;
  }

  displayResult(result) {
    if (result.status === 'complete') {
      console.log(`\nOutput: ${result.email} | ${result.appPassword} | ${result.secretKey}`);
      this.saveResult(`${result.email} | ${result.appPassword} | ${result.secretKey}`);
    } else if (result.status === 'partial' && result.secretKey) {
      console.log(`\nPartial: ${result.email} | ${result.secretKey}`);
      this.saveResult(`${result.email} | PARTIAL | ${result.secretKey}`);
    }
  }

  async saveResult(output) {
    try {
      const outputPath = path.join('output', 'results.txt');
      await fs.appendFile(outputPath, output + '\n');
    } catch (error) {
      // Silent error
    }
  }

  async promptUser() {
    const rl = readline.createInterface({
      input: process.stdin,
      output: process.stdout
    });

    const question = (query) => new Promise(resolve => rl.question(query, resolve));

    const email = await question('Email: ');
    const password = await question('Password: ');

    rl.close();

    return { email, password };
  }

  async promptCSVFile() {
    const rl = readline.createInterface({
      input: process.stdin,
      output: process.stdout
    });

    const question = (query) => new Promise(resolve => rl.question(query, resolve));

    console.log('\nCSV Format: email,password');
    const filePath = await question('CSV file path: ');

    rl.close();

    return filePath.trim();
  }

  async run() {
    console.clear();
    console.log('Google 2FA + App Password Bot');
    console.log('By @xsystemaudit\n');

    await fs.mkdir('output', { recursive: true });
    await fs.mkdir('logs', { recursive: true });

    while (true) {
      console.log('\n1. Single account');
      console.log('2. Bulk (CSV)');
      console.log('3. Exit');

      const rl = readline.createInterface({
        input: process.stdin,
        output: process.stdout
      });

      const choice = await new Promise(resolve => {
        rl.question('\nChoice: ', resolve);
      });
      rl.close();

      if (choice === '1') {
        const { email, password } = await this.promptUser();
        const result = await this.processAccount(email, password);
        this.displayResult(result);

      } else if (choice === '2') {
        const csvPath = await this.promptCSVFile();
        const bulkProcessor = new BulkProcessor();

        try {
          await bulkProcessor.loadCSV(csvPath);
          await bulkProcessor.processAll(this);
        } catch (error) {
          console.log(chalk.red(`Error: ${error.message}`));
        }

      } else if (choice === '3') {
        console.log('\nGoodbye!\n');
        break;
      }

      const continueRl = readline.createInterface({
        input: process.stdin,
        output: process.stdout
      });

      const continueChoice = await new Promise(resolve => {
        continueRl.question('\nPress Enter to continue or q to quit: ', resolve);
      });
      continueRl.close();

      if (continueChoice.toLowerCase() === 'q') {
        break;
      }
    }
  }
}

(async () => {
  const bot = new Google2FABot();
  try {
    await bot.run();
  } catch (error) {
    process.exit(1);
  }
})();

module.exports = Google2FABot;
