#!/bin/bash

# Google 2FA App Password Generator Installer
# Author: @systemaudit
# Enhanced with multi-architecture and fallback support

# Colors
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m'

# Variables
INSTALL_DIR="$HOME/google-2fa-apppassword"
NODE_VERSION="18"
MIN_NODE_VERSION="16"

# Banner
clear
echo -e "${GREEN}"
echo "=========================================="
echo " Google 2FA App Password Generator"
echo " Auto Installer for Linux/VPS"
echo "=========================================="
echo -e "${NC}"

# Function to check if running as root
check_root() {
    if [ "$EUID" -eq 0 ]; then 
        echo -e "${YELLOW}Running as root${NC}"
        SUDO=""
    else
        echo -e "${YELLOW}Running as regular user${NC}"
        SUDO="sudo"
    fi
}

# Function to detect architecture
detect_architecture() {
    ARCH=$(uname -m)
    case $ARCH in
        x86_64)
            ARCH_NAME="x64"
            ;;
        aarch64)
            ARCH_NAME="ARM64"
            ;;
        armv7l|armv6l)
            ARCH_NAME="ARMv7"
            ;;
        *)
            ARCH_NAME="Unknown ($ARCH)"
            ;;
    esac
    echo -e "${BLUE}Detected Architecture: $ARCH_NAME${NC}"
}

# Function to compare versions
version_compare() {
    if [[ $1 == $2 ]]; then
        return 0
    fi
    local IFS=.
    local i ver1=($1) ver2=($2)
    for ((i=${#ver1[@]}; i<${#ver2[@]}; i++)); do
        ver1[i]=0
    done
    for ((i=0; i<${#ver1[@]}; i++)); do
        if [[ -z ${ver2[i]} ]]; then
            ver2[i]=0
        fi
        if ((10#${ver1[i]} > 10#${ver2[i]})); then
            return 1
        fi
        if ((10#${ver1[i]} < 10#${ver2[i]})); then
            return 2
        fi
    done
    return 0
}

# Function to install Node.js via NVM
install_nodejs_nvm() {
    echo -e "${YELLOW}Attempting to install Node.js via NVM...${NC}"
    
    # Check if NVM already installed
    if [ -d "$HOME/.nvm" ]; then
        echo -e "${BLUE}NVM already installed${NC}"
        export NVM_DIR="$HOME/.nvm"
        [ -s "$NVM_DIR/nvm.sh" ] && \. "$NVM_DIR/nvm.sh"
    else
        # Install NVM
        echo -e "${YELLOW}Installing NVM...${NC}"
        curl -o- https://raw.githubusercontent.com/nvm-sh/nvm/v0.39.7/install.sh | bash
        
        # Load NVM
        export NVM_DIR="$HOME/.nvm"
        [ -s "$NVM_DIR/nvm.sh" ] && \. "$NVM_DIR/nvm.sh"
    fi
    
    # Install Node.js
    if command -v nvm &> /dev/null; then
        nvm install $NODE_VERSION
        nvm use $NODE_VERSION
        nvm alias default $NODE_VERSION
        
        # Verify installation
        if command -v node &> /dev/null; then
            echo -e "${GREEN}✓ Node.js $(node -v) installed via NVM${NC}"
            return 0
        fi
    fi
    
    return 1
}

# Function to install Node.js via NodeSource
install_nodejs_nodesource() {
    echo -e "${YELLOW}Attempting to install Node.js via NodeSource...${NC}"
    
    # Download and run NodeSource setup script
    curl -fsSL https://deb.nodesource.com/setup_${NODE_VERSION}.x -o /tmp/nodesource_setup.sh
    $SUDO bash /tmp/nodesource_setup.sh
    rm /tmp/nodesource_setup.sh
    
    # Install Node.js
    $SUDO apt-get install -y nodejs
    
    # Verify installation
    if command -v node &> /dev/null; then
        echo -e "${GREEN}✓ Node.js $(node -v) installed via NodeSource${NC}"
        return 0
    fi
    
    return 1
}

# Function to install Node.js via apt
install_nodejs_apt() {
    echo -e "${YELLOW}Attempting to install Node.js via apt...${NC}"
    
    $SUDO apt-get update
    $SUDO apt-get install -y nodejs npm
    
    # Verify installation
    if command -v node &> /dev/null; then
        echo -e "${GREEN}✓ Node.js $(node -v) installed via apt${NC}"
        return 0
    fi
    
    return 1
}

# Function to install Node.js via n
install_nodejs_n() {
    echo -e "${YELLOW}Attempting to install Node.js via n...${NC}"
    
    # Install n globally
    curl -L https://bit.ly/n-install | bash -s -- -y
    
    # Load n
    export N_PREFIX="$HOME/n"
    export PATH="$N_PREFIX/bin:$PATH"
    
    # Install Node.js
    if command -v n &> /dev/null; then
        n lts
        
        # Verify installation
        if command -v node &> /dev/null; then
            echo -e "${GREEN}✓ Node.js $(node -v) installed via n${NC}"
            return 0
        fi
    fi
    
    return 1
}

# Check root
check_root

# Detect architecture
detect_architecture

# Detect OS
if [ -f /etc/os-release ]; then
    . /etc/os-release
    OS=$ID
    OS_VERSION=$VERSION_ID
else
    echo -e "${RED}Cannot detect OS. Exiting...${NC}"
    exit 1
fi

echo -e "${BLUE}Detected OS: $OS $OS_VERSION${NC}"

# Update package list
echo -e "${YELLOW}Updating package list...${NC}"
$SUDO apt-get update -y 2>&1 | grep -E "packages|upgraded|newly" || echo -e "${GREEN}✓ Package list updated${NC}"

# Install essential packages
echo -e "${YELLOW}Installing essential packages...${NC}"
PACKAGES="curl git wget ca-certificates gnupg unzip"
for pkg in $PACKAGES; do
    if ! command -v $pkg &> /dev/null; then
        echo -e "  Installing $pkg..."
        $SUDO apt-get install -y $pkg 2>&1 | tail -n 1
    fi
done
echo -e "${GREEN}✓ Essential packages installed${NC}"

# Check Node.js
echo -e "${YELLOW}Checking Node.js...${NC}"
NODE_INSTALLED=false

if command -v node &> /dev/null; then
    CURRENT_NODE_VERSION=$(node -v | cut -d'v' -f2 | cut -d'.' -f1)
    echo -e "${BLUE}Current Node.js version: v$(node -v | cut -d'v' -f2)${NC}"
    
    version_compare $CURRENT_NODE_VERSION $MIN_NODE_VERSION
    if [ $? -eq 2 ]; then
        echo -e "${YELLOW}Node.js version is too old. Installing Node.js $NODE_VERSION...${NC}"
        NODE_INSTALLED=false
    else
        echo -e "${GREEN}✓ Node.js version is compatible${NC}"
        NODE_INSTALLED=true
    fi
else
    echo -e "${YELLOW}Node.js not found. Installing Node.js $NODE_VERSION...${NC}"
    NODE_INSTALLED=false
fi

# Install Node.js with fallback methods
if [ "$NODE_INSTALLED" = false ]; then
    echo -e "${YELLOW}Trying multiple installation methods...${NC}"
    
    # Try NVM first (most reliable)
    if install_nodejs_nvm; then
        NODE_INSTALLED=true
    # Try NodeSource
    elif install_nodejs_nodesource; then
        NODE_INSTALLED=true
    # Try apt
    elif install_nodejs_apt; then
        NODE_INSTALLED=true
    # Try n as last resort
    elif install_nodejs_n; then
        NODE_INSTALLED=true
    else
        echo -e "${RED}Failed to install Node.js via all methods!${NC}"
        echo -e "${YELLOW}Please install Node.js manually from: https://nodejs.org${NC}"
        exit 1
    fi
fi

# Verify Node.js and npm
if ! command -v node &> /dev/null; then
    echo -e "${RED}Node.js installation failed!${NC}"
    exit 1
fi

if ! command -v npm &> /dev/null; then
    echo -e "${YELLOW}npm not found. Installing...${NC}"
    $SUDO apt-get install -y npm
fi

echo -e "${GREEN}✓ Node.js $(node -v) ready${NC}"
echo -e "${GREEN}✓ npm $(npm -v) ready${NC}"

# Download or update
echo -e "${YELLOW}Setting up Google 2FA App Password Generator...${NC}"

if [ -d "$INSTALL_DIR" ]; then
    echo -e "${YELLOW}Directory exists. Updating...${NC}"
    
    # Backup old version
    if [ -f "$INSTALL_DIR/.env" ]; then
        echo -e "${BLUE}Backing up .env file...${NC}"
        cp "$INSTALL_DIR/.env" "$INSTALL_DIR/.env.backup"
    fi
    
    # Download latest version
    echo -e "${YELLOW}Downloading latest version...${NC}"
    curl -L https://mirror.malingjenius.ru/project/appassword/latest.zip -o /tmp/appassword.zip
    
    # Remove old files (except .env)
    rm -rf "$INSTALL_DIR"/*
    
    # Extract new version
    unzip -q /tmp/appassword.zip -d "$INSTALL_DIR"
    rm /tmp/appassword.zip
    
    # Restore .env if exists
    if [ -f "$INSTALL_DIR/.env.backup" ]; then
        mv "$INSTALL_DIR/.env.backup" "$INSTALL_DIR/.env"
    fi
    
    echo -e "${GREEN}✓ Updated to latest version${NC}"
else
    echo -e "${YELLOW}Downloading and installing...${NC}"
    
    # Download latest version
    curl -L https://mirror.malingjenius.ru/project/appassword/latest.zip -o /tmp/appassword.zip
    
    # Create install directory
    mkdir -p "$INSTALL_DIR"
    
    # Extract
    unzip -q /tmp/appassword.zip -d "$INSTALL_DIR"
    rm /tmp/appassword.zip
    
    echo -e "${GREEN}✓ Installation complete${NC}"
fi

cd "$INSTALL_DIR"

# Install dependencies
echo -e "${YELLOW}Installing Node.js dependencies...${NC}"
npm install

# Install Playwright and its dependencies
echo -e "${YELLOW}Installing Playwright...${NC}"
npx playwright install chromium

echo -e "${YELLOW}Installing system dependencies for Chromium...${NC}"
$SUDO npx playwright install-deps chromium || npx playwright install-deps chromium

# Create required directories
echo -e "${YELLOW}Creating directories...${NC}"
mkdir -p logs output

# Setup environment file
if [ ! -f .env ]; then
    echo -e "${YELLOW}Creating .env file...${NC}"
    cat > .env << 'EOF'
# Browser Settings
HEADLESS=true
DEBUG_MODE=false

# Timeouts (ms) - Optimized
NAVIGATION_TIMEOUT=15000
IMPLICIT_TIMEOUT=6000
VERIFICATION_TIMEOUT=8000

# Processing - Optimized
BATCH_SIZE=4
BATCH_DELAY=3000

# TOTP API
TOTP_API_URL=https://2fa.live/tok

# Output
OUTPUT_FORMAT=csv
OUTPUT_PATH=output/results.csv
EOF
    echo -e "${GREEN}✓ Created .env file${NC}"
fi

# Create start script
echo -e "${YELLOW}Creating start script...${NC}"
cat > "$INSTALL_DIR/start.sh" << 'EOF'
#!/bin/bash
cd "$(dirname "$0")"

# Load NVM if available
export NVM_DIR="$HOME/.nvm"
[ -s "$NVM_DIR/nvm.sh" ] && \. "$NVM_DIR/nvm.sh"

# Load n if available
export N_PREFIX="$HOME/n"
export PATH="$N_PREFIX/bin:$PATH"

npm start
EOF
chmod +x "$INSTALL_DIR/start.sh"

# Add to PATH (optional)
if ! grep -q "google-2fa-apppassword" ~/.bashrc; then
    echo -e "${YELLOW}Adding to PATH...${NC}"
    echo "alias google2fa='cd $INSTALL_DIR && ./start.sh'" >> ~/.bashrc
    echo -e "${GREEN}✓ Added 'google2fa' command${NC}"
fi

# Success message
echo -e "${GREEN}"
echo "=========================================="
echo " Installation Complete!"
echo "=========================================="
echo -e "${NC}"
echo -e "${BLUE}Installation Directory:${NC} $INSTALL_DIR"
echo -e "${BLUE}Architecture:${NC} $ARCH_NAME"
echo ""
echo -e "${GREEN}Next Steps:${NC}"
echo -e "1. ${YELLOW}Edit configuration (optional):${NC}"
echo -e "   nano $INSTALL_DIR/.env"
echo ""
echo -e "2. ${YELLOW}Run the application:${NC}"
echo -e "   ${GREEN}Option A:${NC} cd $INSTALL_DIR && npm start"
echo -e "   ${GREEN}Option B:${NC} $INSTALL_DIR/start.sh"
echo -e "   ${GREEN}Option C:${NC} google2fa ${BLUE}(after reloading shell)${NC}"
echo ""
echo -e "3. ${YELLOW}Reload shell for alias:${NC}"
echo -e "   source ~/.bashrc"
echo ""

# Ask if user wants to start now
echo -e "${YELLOW}Do you want to start the application now? (y/n)${NC}"
read -n 1 -r
echo
if [[ $REPLY =~ ^[Yy]$ ]]; then
    echo -e "${GREEN}Starting Google 2FA App Password Generator...${NC}"
    cd "$INSTALL_DIR"
    npm start
else
    echo -e "${BLUE}You can start the application later using:${NC}"
    echo -e "cd $INSTALL_DIR && npm start"
fi
